<?php
namespace CpExtended;

class Api
{
    /**
     * Build Authorization header for WHM/cPanel.
     * Preferred: WHM API Token stored in serverpassword (recommended).
     * Fallback: accesshash (serveraccesshash).
     */
    public static function authHeader(array $params): string
    {
        $user = (string)($params['serverusername'] ?? 'root');
        $token = (string)($params['serverpassword'] ?? ''); // store API token here
        $hash  = (string)($params['serveraccesshash'] ?? '');

        if ($token !== '') {
            // WHM API Token format: Authorization: whm user:token
            return 'Authorization: whm ' . $user . ':' . $token;
        }

        // Access hash can contain newlines/spaces; normalize
        $hash = preg_replace('/\s+/', '', $hash);
        if ($hash !== '') {
            return 'Authorization: WHM ' . $user . ':' . $hash;
        }

        return '';
    }

    public static function baseUrl(array $params): string
    {
        $host = (string)($params['serverhostname'] ?? $params['serverip'] ?? '');
        if ($host === '') $host = '127.0.0.1';
        $secure = !empty($params['serversecure']) && (string)$params['serversecure'] !== '0';
        $scheme = $secure ? 'https' : 'http';
        $port = (int)($secure ? ($params['serverport'] ?? 2087) : ($params['serverport'] ?? 2086));
        return $scheme . '://' . $host . ':' . $port;
    }

    /**
     * Call WHM JSON API endpoint on 2087/2086.
     * $path should start with /json-api/...
     */
    public static function whm(array $params, string $path, array $query = []): array
    {
        $url = self::baseUrl($params) . $path;
        if (!empty($query)) {
            $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($query);
        }

        $headers = [];
        $auth = self::authHeader($params);
        if ($auth) $headers[] = $auth;

        return self::curlJson($url, $headers);
    }

    /**
     * Create a user session URL (SSO) for cPanel / Webmail.
     * service: cpaneld | webmaild
     */
    public static function createUserSession(array $params, string $cpUser, string $service = 'cpaneld', string $gotoUri = ''): ?string
    {
        $query = [
            'api.version' => 1,
            'user' => $cpUser,
            'service' => $service,
        ];
        if ($gotoUri !== '') {
            $query['goto_uri'] = $gotoUri;
        }
        $res = self::whm($params, '/json-api/create_user_session', $query);

        // Expected: data.url
        $url = $res['data']['url'] ?? $res['data']['session']['url'] ?? null;
        if (is_string($url) && $url !== '') return $url;

        return null;
    }

    /**
     * Call a cPanel API module/function (legacy wrapper) via WHM endpoint /json-api/cpanel.
     * This works without a per-user token because WHM authenticates and impersonates cPanel user.
     */
    public static function cpanelApi(array $params, string $cpUser, string $module, string $func, array $args = []): array
    {
        $query = array_merge([
            'api.version' => 1,
            'cpanel_jsonapi_user' => $cpUser,
            'cpanel_jsonapi_module' => $module,
            'cpanel_jsonapi_func' => $func,
        ], $args);

        return self::whm($params, '/json-api/cpanel', $query);
    }

    private static function curlJson(string $url, array $headers = []): array
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }

        $body = curl_exec($ch);
        $err  = curl_error($ch);
        $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($body === false) {
            return ['ok' => false, 'http_code' => $code, 'error' => $err ?: 'curl_error', 'raw' => null];
        }

        $decoded = json_decode($body, true);
        if (!is_array($decoded)) {
            return ['ok' => false, 'http_code' => $code, 'error' => 'invalid_json', 'raw' => $body];
        }

        return $decoded + ['ok' => ($code >= 200 && $code < 300), 'http_code' => $code];
    }
}
