<?php
/**
 * WHMCS Server Module: cPanel Extended (Skeleton)
 * Part 8: Provisioning stubs + ClientArea shell
 *
 * IMPORTANT:
 * - Real provisioning + cPanel/WHM API actions will be implemented in the next iteration.
 * - This skeleton proves structure and reads per-product feature toggles from the addon module tables.
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

require_once __DIR__ . '/lib/Settings.php';
require_once __DIR__ . '/lib/Api.php';
require_once __DIR__ . '/lib/Utils.php';
require_once __DIR__ . '/lib/Queue.php';

function cp_extended_MetaData()
{
    return [
        'DisplayName' => 'cPanel Extended',
        'APIVersion' => '1.1',
        'RequiresServer' => true,
        'DefaultNonSSLPort' => '2086',
        'DefaultSSLPort' => '2087',
        'ServiceSingleSignOnLabel' => 'Login to cPanel',
        'AdminSingleSignOnLabel' => 'Login to WHM',
    ];
}

function cp_extended_ConfigOptions()
{
    return [
        'Package Name' => [
            'Type' => 'text',
            'Size' => '25',
            'Description' => 'cPanel package name to provision',
        ],
        'Enable Extended Panel' => [
            'Type' => 'yesno',
            'Description' => 'Show Extended UI in client area (overrides addon product setting)',
            'Default' => '',
        ],
    ];
}

// ---- Provisioning stubs ----
function cp_extended_CreateAccount($params)
{
    // TODO Part 9+: call WHM API createacct using server credentials/token
    return "success";
}

function cp_extended_SuspendAccount($params)
{
    // TODO: call WHM suspendacct
    return "success";
}

function cp_extended_UnsuspendAccount($params)
{
    // TODO: call WHM unsuspendacct
    return "success";
}

function cp_extended_TerminateAccount($params)
{
    // TODO: call WHM removeacct
    return "success";
}

function cp_extended_ChangePackage($params)
{
    // TODO: call WHM changepackage / modifyacct
    return "success";
}

function cp_extended_ChangePassword($params)
{
    // TODO: call WHM passwd
    return "success";
}

/**
 * Client Area output (the real “Extended” UI belongs here).
 * In Part 8 we provide a shell that reads per-product toggles from the addon module.
 */
function cp_extended_ClientArea($params)
{
    $serviceId = (int)($params['serviceid'] ?? 0);
    $productId = (int)($params['pid'] ?? 0);

    $settings = CpExtended\Settings::forProduct($productId);

    // Allow config option override (per-service/module config)
    $override = (string)($params['configoption2'] ?? '');
    if ($override === 'on') {
        $settings['enabled'] = true;
    } elseif ($override === 'off') {
        $settings['enabled'] = false;
    }

    // cPanel username is typically in $params['username']
    $cpUser = (string)($params['username'] ?? '');

    // Routing
    $page = CpExtended\Utils::safePage((string)($_GET['cpxt_page'] ?? 'home'));


    $flash = '';
    $flashType = 'info';

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $settings['enabled'] && $cpUser !== '') {
        $act = CpExtended\Utils::safePage((string)($_POST['cpxt_action'] ?? ''));
        try {
            if ($act === 'email_create') {
                $local = trim((string)($_POST['local'] ?? ''));
                $emailDomain = trim((string)($_POST['email_domain'] ?? ''));
                $password = (string)($_POST['password'] ?? '');
                $quota = (string)($_POST['quota'] ?? '0');
                if ($local === '' || $emailDomain === '' || $password === '') throw new \Exception('Missing required fields.');
                $email = $local . '@' . $emailDomain;

                $taskId = CpExtended\Queue::enqueue('email_create', ['email'=>$email,'password'=>$password,'quota'=>$quota], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: create email (' . $email . '). Task #' . $taskId;
            }

            if ($act === 'email_delete') {
                $email = trim((string)($_POST['email'] ?? ''));
                if ($email === '') throw new \Exception('Missing email.');
                $taskId = CpExtended\Queue::enqueue('email_delete', ['email'=>$email], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: delete email (' . $email . '). Task #' . $taskId;
            }

            if ($act === 'email_bulk_delete') {
                $emails = $_POST['emails'] ?? [];
                if (!is_array($emails) || count($emails) === 0) throw new \Exception('Select at least one email.');
                $count = 0;
                foreach ($emails as $email) {
                    $email = trim((string)$email);
                    if ($email === '') continue;
                    CpExtended\Queue::enqueue('email_delete', ['email'=>$email], (int)($params['userid']??0), $serviceId);
                    $count++;
                }
                $flash = 'Queued: delete ' . $count . ' email account(s).';
            }

            if ($act === 'email_passwd') {
                $email = trim((string)($_POST['email'] ?? ''));
                $password = (string)($_POST['password'] ?? '');
                if ($email === '' || $password === '') throw new \Exception('Missing required fields.');
                $taskId = CpExtended\Queue::enqueue('email_passwd', ['email'=>$email,'password'=>$password], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: update password (' . $email . '). Task #' . $taskId;
            }

            if ($act === 'email_quota') {
                $email = trim((string)($_POST['email'] ?? ''));
                $quota = (string)($_POST['quota'] ?? '0');
                if ($email === '') throw new \Exception('Missing email.');
                $taskId = CpExtended\Queue::enqueue('email_quota', ['email'=>$email,'quota'=>$quota], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: update quota (' . $email . '). Task #' . $taskId;
            }

            if ($act === 'ftp_create') {
                $user = trim((string)($_POST['user'] ?? ''));
                $password = (string)($_POST['password'] ?? '');
                $quota = (string)($_POST['quota'] ?? '0');
                $homedir = (string)($_POST['homedir'] ?? '/');
                if ($user === '' || $password === '') throw new \Exception('Missing required fields.');
                $taskId = CpExtended\Queue::enqueue('ftp_create', ['user'=>$user,'password'=>$password,'quota'=>$quota,'homedir'=>$homedir], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: create FTP (' . $user . '). Task #' . $taskId;
            }

            if ($act === 'ftp_delete') {
                $user = trim((string)($_POST['user'] ?? ''));
                $destroy = isset($_POST['destroy']) ? '1' : '0';
                if ($user === '') throw new \Exception('Missing ftp user.');
                $taskId = CpExtended\Queue::enqueue('ftp_delete', ['user'=>$user,'destroy'=>$destroy], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: delete FTP (' . $user . '). Task #' . $taskId;
            }

            if ($act === 'ftp_passwd') {
                $user = trim((string)($_POST['user'] ?? ''));
                $password = (string)($_POST['password'] ?? '');
                if ($user === '' || $password === '') throw new \Exception('Missing required fields.');
                $taskId = CpExtended\Queue::enqueue('ftp_passwd', ['user'=>$user,'password'=>$password], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: update FTP password (' . $user . '). Task #' . $taskId;
            }

            if ($act === 'cron_add') {
                $line = trim((string)($_POST['line'] ?? ''));
                if ($line === '') throw new \Exception('Missing cron line.');
                $taskId = CpExtended\Queue::enqueue('cron_add', ['line'=>$line], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: add cron. Task #' . $taskId;
            }

            if ($act === 'cron_delete') {
                $linekey = trim((string)($_POST['linekey'] ?? ''));
                if ($linekey === '') throw new \Exception('Missing linekey.');
                $taskId = CpExtended\Queue::enqueue('cron_delete', ['linekey'=>$linekey], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: delete cron. Task #' . $taskId;
            }

            if ($act === 'mysql_db_create') {
                $db = trim((string)($_POST['db'] ?? ''));
                if ($db === '') throw new \Exception('Missing db name.');
                $taskId = CpExtended\Queue::enqueue('mysql_db_create', ['db'=>$db], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: create DB (' . $db . '). Task #' . $taskId;
            }

            if ($act === 'mysql_db_delete') {
                $db = trim((string)($_POST['db'] ?? ''));
                if ($db === '') throw new \Exception('Missing db name.');
                $taskId = CpExtended\Queue::enqueue('mysql_db_delete', ['db'=>$db], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: delete DB (' . $db . '). Task #' . $taskId;
            }

            if ($act === 'mysql_user_create') {
                $user = trim((string)($_POST['user'] ?? ''));
                $password = (string)($_POST['password'] ?? '');
                if ($user === '' || $password === '') throw new \Exception('Missing user/password.');
                $taskId = CpExtended\Queue::enqueue('mysql_user_create', ['user'=>$user,'password'=>$password], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: create DB user (' . $user . '). Task #' . $taskId;
            }

            if ($act === 'mysql_user_delete') {
                $user = trim((string)($_POST['user'] ?? ''));
                if ($user === '') throw new \Exception('Missing user.');
                $taskId = CpExtended\Queue::enqueue('mysql_user_delete', ['user'=>$user], (int)($params['userid']??0), $serviceId);
                $flash = 'Queued: delete DB user (' . $user . '). Task #' . $taskId;
            }

        } catch (\Throwable $e) {
            $flashType = 'error';
            $flash = $e->getMessage();
        }
    }

    // SSO actions (requires WHM API token / access hash configured on server)
    $sso = CpExtended\Utils::safePage((string)($_GET['cpxt_sso'] ?? ''));
    if ($sso && $settings['enabled'] && $cpUser !== '') {
        $goto = '';
        $service = 'cpaneld';
        if ($sso === 'webmail') $service = 'webmaild';
        if ($sso === 'filemanager') { $service = 'cpaneld'; $goto = '/frontend/jupiter/filemanager/index.html'; }
        if ($sso === 'phpmyadmin') { $service = 'cpaneld'; $goto = '/3rdparty/phpMyAdmin/index.php'; }

        $url = CpExtended\Api::createUserSession($params, $cpUser, $service, $goto);
        if ($url) {
            return [
                'templatefile' => 'redirect',
                'vars' => [
                    'redirectUrl' => $url,
                ],
            ];
        }

        return [
            'templatefile' => 'error',
            'vars' => [
                'errorTitle' => 'SSO Failed',
                'errorMessage' => 'Unable to create a login session. Please contact support.',
            ],
        ];
    }

    // Feature: email accounts (list)
    $emailAccounts = [];
    $emailError = '';
    if ($page === 'email' && $settings['enabled'] && $cpUser !== '') {
        // Uses WHM wrapper endpoint to execute Email::list_pops (commonly available)
        $res = CpExtended\Api::cpanelApi($params, $cpUser, 'Email', 'list_pops', [
            'api.version' => 1,
        ]);

        // Fallback for some servers: listpops
        if (!isset($res['cpanelresult']['data']) && !isset($res['data'])) {
            $res = CpExtended\Api::cpanelApi($params, $cpUser, 'Email', 'listpops', [
                'api.version' => 1,
            ]);
        }

        // Try to normalize different response shapes
        if (isset($res['cpanelresult']['data']) && is_array($res['cpanelresult']['data'])) {
            $emailAccounts = $res['cpanelresult']['data'];
        } elseif (isset($res['data']) && is_array($res['data'])) {
            $emailAccounts = $res['data'];
        } else {
            $emailError = 'Failed to load email accounts.';
        }
    }

    

    // Feature: FTP accounts (list)
    $ftpAccounts = [];
    $ftpError = '';
    if ($page === 'ftp' && $settings['enabled'] && $cpUser !== '') {
        $res = CpExtended\Api::cpanelApi($params, $cpUser, 'Ftp', 'listftp', [
            'api.version' => 1,
        ]);
        if (isset($res['cpanelresult']['data']) && is_array($res['cpanelresult']['data'])) {
            $ftpAccounts = $res['cpanelresult']['data'];
        } elseif (isset($res['data']) && is_array($res['data'])) {
            $ftpAccounts = $res['data'];
        } else {
            $ftpError = 'Failed to load FTP accounts.';
        }
    }

    // Feature: Cron jobs (list)
    $cronItems = [];
    $cronError = '';
    if ($page === 'cron' && $settings['enabled'] && $cpUser !== '') {
        $res = CpExtended\Api::cpanelApi($params, $cpUser, 'Cron', 'listcron', [
            'api.version' => 1,
        ]);
        if (isset($res['cpanelresult']['data']) && is_array($res['cpanelresult']['data'])) {
            $cronItems = $res['cpanelresult']['data'];
        } elseif (isset($res['data']) && is_array($res['data'])) {
            $cronItems = $res['data'];
        } else {
            $cronError = 'Failed to load cron jobs.';
        }
    }

    // Feature: MySQL (list dbs/users)
    $mysqlDbs = [];
    $mysqlUsers = [];
    $mysqlError = '';
    if ($page === 'mysql' && $settings['enabled'] && $cpUser !== '') {
        $res1 = CpExtended\Api::cpanelApi($params, $cpUser, 'Mysql', 'listdbs', [
            'api.version' => 1,
        ]);
        if (isset($res1['cpanelresult']['data']) && is_array($res1['cpanelresult']['data'])) {
            $mysqlDbs = $res1['cpanelresult']['data'];
        } elseif (isset($res1['data']) && is_array($res1['data'])) {
            $mysqlDbs = $res1['data'];
        }

        $res2 = CpExtended\Api::cpanelApi($params, $cpUser, 'Mysql', 'listusers', [
            'api.version' => 1,
        ]);
        if (isset($res2['cpanelresult']['data']) && is_array($res2['cpanelresult']['data'])) {
            $mysqlUsers = $res2['cpanelresult']['data'];
        } elseif (isset($res2['data']) && is_array($res2['data'])) {
            $mysqlUsers = $res2['data'];
        }

        if (count($mysqlDbs) === 0 && count($mysqlUsers) === 0) {
            // don't hard-fail; some servers return empty until created
        }
    }

$vars = [
        'serviceId' => $serviceId,
        'domain' => (string)($params['domain'] ?? ''),
        'productId' => $productId,
        'cpUser' => $cpUser,
        'enabled' => (bool)$settings['enabled'],
        'template' => (string)$settings['template'],
        'features' => (array)$settings['features'],
        'installer' => (string)$settings['installer'],
        'page' => $page,
        'emailAccounts' => $emailAccounts,
        'ftpAccounts' => $ftpAccounts,
        'ftpError' => $ftpError,
        'cronItems' => $cronItems,
        'cronError' => $cronError,
        'mysqlDbs' => $mysqlDbs,
        'mysqlUsers' => $mysqlUsers,
        'mysqlError' => $mysqlError,
        'emailError' => $emailError,
        'flash' => $flash,
        'flashType' => $flashType,
        'tasks' => CpExtended\Queue::latestForService($serviceId, 10),
        'baseLink' => 'clientarea.php?action=productdetails&id=' . $serviceId,
    ];

    return [
        'templatefile' => 'clientarea',
        'breadcrumb' => [
            'index.php?m=cp_extended' => 'cPanel Extended',
        ],
        'vars' => $vars,
    ];
}

