<?php
/**
 * cPanel Extended Admin - Queue Worker Endpoint (Part 5)
 *
 * Security:
 * - Runs in WHMCS context
 * - Requires a secret token stored in global settings key: queue.worker_secret
 * - Accepts GET/POST: ?token=...&limit=...
 *
 * Usage (cron):
 * php -q /path/to/whmcs/modules/addons/cp_extended_admin/queue_worker.php token=YOURTOKEN limit=10
 * OR web call (if accessible): https://yourwhmcs/admin/modules/addons/.../queue_worker.php?token=...
 */

define('WHMCS_ROOT', dirname(__DIR__, 3)); // modules/addons/cp_extended_admin -> whmcs root
require_once WHMCS_ROOT . '/init.php';

use WHMCS\Database\Capsule;

require_once __DIR__ . '/lib/helpers.php';

// Use server module API client for task execution
$serverApiPath = WHMCS_ROOT . '/modules/servers/cp_extended/lib/Api.php';
if (file_exists($serverApiPath)) {
    require_once $serverApiPath;
}


function cpExtAdmin_build_service_params(int $serviceId): array
{
    try {
        $svc = Capsule::table('tblhosting')->where('id', $serviceId)->first();
        if (!$svc) return [];

        $server = Capsule::table('tblservers')->where('id', (int)$svc->server)->first();
        if (!$server) return [];

        $serverPass = (string)($server->password ?? '');
        if (function_exists('decrypt')) {
            $serverPass = decrypt($serverPass);
        }

        $accessHash = (string)($server->accesshash ?? '');
        if (function_exists('decrypt')) {
            $try = decrypt($accessHash);
            if (is_string($try) && $try !== '') $accessHash = $try;
        }

        return [
            'serviceid' => (int)$svc->id,
            'pid' => (int)$svc->packageid,
            'userid' => (int)$svc->userid,
            'domain' => (string)$svc->domain,
            'username' => (string)$svc->username,
            'serverid' => (int)$server->id,
            'serverhostname' => (string)$server->hostname,
            'serverip' => (string)$server->ipaddress,
            'serverusername' => (string)$server->username,
            'serverpassword' => (string)$serverPass,
            'serveraccesshash' => (string)$accessHash,
            'serversecure' => (string)($server->secure ?? 'on') === 'on' ? 1 : 0,
            'serverport' => (int)($server->port ?? 2087),
        ];
    } catch (\Throwable $e) {
        return [];
    }
}

function cpExtAdmin_param($key, $default=null) {
    // CLI support: php file.php token=... limit=...
    global $argv;
    if (php_sapi_name() === 'cli' && isset($argv) && is_array($argv)) {
        foreach ($argv as $arg) {
            if (strpos($arg, '=') !== false) {
                [$k,$v] = explode('=', $arg, 2);
                if ($k === $key) return $v;
            }
        }
    }
    if (isset($_REQUEST[$key])) return $_REQUEST[$key];
    return $default;
}

$token = (string)cpExtAdmin_param('token', '');
$limit = (int)cpExtAdmin_param('limit', 5);
if ($limit < 1) $limit = 1;
if ($limit > 25) $limit = 25;

$secret = (string)cpExtAdmin_setting_get('queue.worker_secret', 'global', '');
if ($secret === '' || !hash_equals($secret, $token)) {
    http_response_code(403);
    echo "Forbidden\n";
    exit;
}

// Lock token for this run
$lockToken = bin2hex(random_bytes(16));
$now = cpExtAdmin_now();

$processed = 0;
$results = [];

try {
    // Fetch pending tasks that are available and not locked
    $tasks = Capsule::table('mod_cp_ext_queue')
        ->where('status', 'pending')
        ->where(function($q){
            $q->whereNull('available_at')->orWhere('available_at', '<=', cpExtAdmin_now());
        })
        ->whereNull('locked_at')
        ->orderBy('id', 'asc')
        ->limit($limit)
        ->get();

    foreach ($tasks as $task) {
        // Attempt lock (optimistic)
        $updated = Capsule::table('mod_cp_ext_queue')
            ->where('id', $task->id)
            ->whereNull('locked_at')
            ->update([
                'locked_at' => $now,
                'lock_token' => $lockToken,
                'status' => 'running',
                'updated_at' => $now,
            ]);

        if (!$updated) {
            continue; // someone else took it
        }

        $processed++;
        $payload = [];
        if (!empty($task->payload_json)) {
            $tmp = json_decode((string)$task->payload_json, true);
            if (is_array($tmp)) $payload = $tmp;
        }

        // Dispatch task types
        $type = (string)$task->type;
        $ok = true;
        $note = '';
        $out = null;

        try {
            if (in_array($type, ['email_create','email_delete','email_passwd','email_quota','ftp_create','ftp_delete','ftp_passwd','cron_add','cron_delete','mysql_db_create','mysql_db_delete','mysql_user_create','mysql_user_delete'], true)) {
                if (!class_exists('CpExtended\\Api')) {
                    throw new \Exception('Server module API client not found. Upload /modules/servers/cp_extended/');
                }

                $serviceId = (int)($task->related_service_id ?? 0);
                $params = cpExtAdmin_build_service_params($serviceId);
                if (empty($params)) throw new \Exception('Invalid service/server context.');
                $cpUser = (string)($params['username'] ?? '');
                if ($cpUser === '') throw new \Exception('Missing cPanel username.');

                $email = (string)($payload['email'] ?? '');
                if ($email === '') throw new \Exception('Missing email.');

                if ($type === 'email_create') {
                    $password = (string)($payload['password'] ?? '');
                    $quota = (string)($payload['quota'] ?? '0');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Email', 'addpop', [
                        'email' => $email,
                        'password' => $password,
                        'quota' => $quota,
                    ]);
                    $out = $resp;
                    $note = 'Email created';
                } elseif ($type === 'email_delete') {
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Email', 'delpop', [
                        'email' => $email,
                    ]);
                    $out = $resp;
                    $note = 'Email deleted';
                } elseif ($type === 'email_passwd') {
                    $password = (string)($payload['password'] ?? '');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Email', 'passwdpop', [
                        'email' => $email,
                        'password' => $password,
                    ]);
                    $out = $resp;
                    $note = 'Password updated';
                } elseif ($type === 'email_quota') {
                    $quota = (string)($payload['quota'] ?? '0');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Email', 'editquota', [
                        'email' => $email,
                        'quota' => $quota,
                    ]);
                    $out = $resp;
                    $note = 'Quota updated';
                }
            
            } elseif (in_array($type, ['ftp_create','ftp_delete','ftp_passwd'], true)) {
                $serviceId = (int)($task->related_service_id ?? 0);
                $params = cpExtAdmin_build_service_params($serviceId);
                if (empty($params)) throw new \Exception('Invalid service/server context.');
                $cpUser = (string)($params['username'] ?? '');
                if ($cpUser === '') throw new \Exception('Missing cPanel username.');

                $user = (string)($payload['user'] ?? '');
                if ($user === '') throw new \Exception('Missing ftp user.');
                $password = (string)($payload['password'] ?? '');

                if ($type === 'ftp_create') {
                    $quota = (string)($payload['quota'] ?? '0');
                    $homedir = (string)($payload['homedir'] ?? '/');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Ftp', 'addftp', [
                        'user' => $user,
                        'pass' => $password,
                        'quota' => $quota,
                        'homedir' => $homedir,
                    ]);
                    $out = $resp;
                    $note = 'FTP created';
                } elseif ($type === 'ftp_delete') {
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Ftp', 'delftp', [
                        'user' => $user,
                        'destroy' => (string)($payload['destroy'] ?? '0'),
                    ]);
                    $out = $resp;
                    $note = 'FTP deleted';
                } elseif ($type === 'ftp_passwd') {
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Ftp', 'passwdftp', [
                        'user' => $user,
                        'pass' => $password,
                    ]);
                    $out = $resp;
                    $note = 'FTP password updated';
                }

            } elseif (in_array($type, ['cron_add','cron_delete'], true)) {
                $serviceId = (int)($task->related_service_id ?? 0);
                $params = cpExtAdmin_build_service_params($serviceId);
                if (empty($params)) throw new \Exception('Invalid service/server context.');
                $cpUser = (string)($params['username'] ?? '');
                if ($cpUser === '') throw new \Exception('Missing cPanel username.');

                if ($type === 'cron_add') {
                    $line = (string)($payload['line'] ?? '');
                    if ($line === '') throw new \Exception('Missing cron line.');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Cron', 'add_line', [
                        'line' => $line,
                    ]);
                    $out = $resp;
                    $note = 'Cron added';
                } elseif ($type === 'cron_delete') {
                    $linekey = (string)($payload['linekey'] ?? '');
                    if ($linekey === '') throw new \Exception('Missing linekey.');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Cron', 'remove_line', [
                        'linekey' => $linekey,
                    ]);
                    $out = $resp;
                    $note = 'Cron deleted';
                }

            } elseif (in_array($type, ['mysql_db_create','mysql_db_delete','mysql_user_create','mysql_user_delete'], true)) {
                $serviceId = (int)($task->related_service_id ?? 0);
                $params = cpExtAdmin_build_service_params($serviceId);
                if (empty($params)) throw new \Exception('Invalid service/server context.');
                $cpUser = (string)($params['username'] ?? '');
                if ($cpUser === '') throw new \Exception('Missing cPanel username.');

                if ($type === 'mysql_db_create') {
                    $db = (string)($payload['db'] ?? '');
                    if ($db === '') throw new \Exception('Missing db name.');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Mysql', 'adddb', [
                        'db' => $db,
                    ]);
                    $out = $resp;
                    $note = 'MySQL DB created';
                } elseif ($type === 'mysql_db_delete') {
                    $db = (string)($payload['db'] ?? '');
                    if ($db === '') throw new \Exception('Missing db name.');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Mysql', 'deldb', [
                        'db' => $db,
                    ]);
                    $out = $resp;
                    $note = 'MySQL DB deleted';
                } elseif ($type === 'mysql_user_create') {
                    $user = (string)($payload['user'] ?? '');
                    $pass = (string)($payload['password'] ?? '');
                    if ($user === '' || $pass === '') throw new \Exception('Missing user/password.');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Mysql', 'adduser', [
                        'user' => $user,
                        'pass' => $pass,
                    ]);
                    $out = $resp;
                    $note = 'MySQL user created';
                } elseif ($type === 'mysql_user_delete') {
                    $user = (string)($payload['user'] ?? '');
                    if ($user === '') throw new \Exception('Missing user.');
                    $resp = \CpExtended\Api::cpanelApi($params, $cpUser, 'Mysql', 'deluser', [
                        'user' => $user,
                    ]);
                    $out = $resp;
                    $note = 'MySQL user deleted';
                }
} else {
                $note = 'No-op (unknown type)';
            }
        } catch (\Throwable $e) {
            $ok = false;
            $note = 'Error: ' . $e->getMessage();
            $out = ['error' => $e->getMessage()];
        }

        $result = [
            'ok' => $ok,
            'note' => $note,
            'type' => $type,
            'payload' => $payload,
            'output' => $out,
            'ran_at' => $now,
        ];

        if ($ok) {
            Capsule::table('mod_cp_ext_queue')
                ->where('id', $task->id)
                ->where('lock_token', $lockToken)
                ->update([
                    'status' => 'done',
                    'result_json' => json_encode($result, JSON_UNESCAPED_SLASHES),
                    'locked_at' => null,
                    'lock_token' => null,
                    'updated_at' => cpExtAdmin_now(),
                ]);
        } else {
            Capsule::table('mod_cp_ext_queue')
                ->where('id', $task->id)
                ->where('lock_token', $lockToken)
                ->update([
                    'status' => 'failed',
                    'attempts' => (int)$task->attempts + 1,
                    'result_json' => json_encode($result, JSON_UNESCAPED_SLASHES),
                    'locked_at' => null,
                    'lock_token' => null,
                    'updated_at' => cpExtAdmin_now(),
                ]);
        }

        cpExtAdmin_log($ok ? 'info' : 'error', 'Queue task processed', ['task_id' => (int)$task->id, 'type' => $type, 'ok' => $ok]);

        $results[] = ['id' => (int)$task->id, 'type' => $type, 'status' => $ok ? 'done' : 'failed'];
    }

    header('Content-Type: application/json');
    echo json_encode([
        'ok' => true,
        'processed' => $processed,
        'results' => $results,
    ], JSON_UNESCAPED_SLASHES);

} catch (\Throwable $e) {
    http_response_code(500);
    cpExtAdmin_log('error', 'Queue worker error', ['error' => $e->getMessage()]);
    echo "Error: " . $e->getMessage() . "\n";
}
