<?php
/**
 * Helper utilities for cPanel Extended Admin addon module.
 */
use WHMCS\Database\Capsule;

if (!function_exists('cpExtAdmin_now')) {
    function cpExtAdmin_now(): string
    {
        return gmdate('Y-m-d H:i:s');
    }
}

if (!function_exists('cpExtAdmin_tableExists')) {
    function cpExtAdmin_tableExists(string $table): bool
    {
        return Capsule::schema()->hasTable($table);
    }
}

if (!function_exists('cpExtAdmin_setting_get')) {
    /**
     * Get a setting value by key for a scope (global, product:<id>, server:<id>)
     */
    function cpExtAdmin_setting_get(string $key, string $scope = 'global', $default = null)
    {
        $row = Capsule::table('mod_cp_ext_settings')
            ->where('scope', $scope)
            ->where('key', $key)
            ->first();
        if (!$row) return $default;
        return $row->value;
    }
}

if (!function_exists('cpExtAdmin_setting_set')) {
    /**
     * Set a setting value by key for a scope.
     */
    function cpExtAdmin_setting_set(string $key, string $value, string $scope = 'global'): void
    {
        $existing = Capsule::table('mod_cp_ext_settings')
            ->where('scope', $scope)
            ->where('key', $key)
            ->first();

        if ($existing) {
            Capsule::table('mod_cp_ext_settings')
                ->where('id', $existing->id)
                ->update([
                    'value' => $value,
                    'updated_at' => cpExtAdmin_now(),
                ]);
        } else {
            Capsule::table('mod_cp_ext_settings')->insert([
                'scope' => $scope,
                'key' => $key,
                'value' => $value,
                'created_at' => cpExtAdmin_now(),
                'updated_at' => cpExtAdmin_now(),
            ]);
        }
    }
}

if (!function_exists('cpExtAdmin_log')) {
    /**
     * Write to module log table.
     */
    function cpExtAdmin_log(string $level, string $message, array $context = []): void
    {
        try {
            Capsule::table('mod_cp_ext_logs')->insert([
                'level' => substr($level, 0, 12),
                'message' => $message,
                'context_json' => json_encode($context, JSON_UNESCAPED_SLASHES),
                'created_at' => cpExtAdmin_now(),
            ]);
        } catch (\Throwable $e) {
            // If logging fails, swallow to avoid breaking admin UI
        }
    }
}

if (!function_exists('cpExtAdmin_admin_base_url')) {
    function cpExtAdmin_admin_base_url(): string
    {
        // WHMCS admin addon routing uses: addonmodules.php?module=<name>
        return 'addonmodules.php?module=cp_extended_admin';
    }
}

if (!function_exists('cpExtAdmin_csrf_field')) {
    function cpExtAdmin_csrf_field(): string
    {
        // WHMCS includes a CSRF token in admin as $token sometimes; safest: use generate_token()
        if (function_exists('generate_token')) {
            return generate_token('plain');
        }
        return '';
    }
}

if (!function_exists('cpExtAdmin_require_admin')) {
    /**
     * Enforce admin permission for this module.
     * WHMCS will handle access control if the role permission is not granted,
     * but we keep this for defense-in-depth and future expansion.
     */
    function cpExtAdmin_require_admin(): void
    {
        if (!defined('ADMINAREA')) return;
        // Most enforcement is handled by WHMCS; additional checks can be added later.
    }
}


if (!function_exists('cpExtAdmin_queue_enqueue')) {
    /**
     * Enqueue a task into mod_cp_ext_queue.
     */
    function cpExtAdmin_queue_enqueue(string $type, array $payload = [], int $userId = null, int $serviceId = null, string $availableAt = null): int
    {
        $id = Capsule::table('mod_cp_ext_queue')->insertGetId([
            'type' => $type,
            'status' => 'pending',
            'related_user_id' => $userId,
            'related_service_id' => $serviceId,
            'attempts' => 0,
            'available_at' => $availableAt,
            'payload_json' => json_encode($payload, JSON_UNESCAPED_SLASHES),
            'created_at' => cpExtAdmin_now(),
            'updated_at' => cpExtAdmin_now(),
        ]);
        return (int)$id;
    }
}


if (!function_exists('cpExtAdmin_t_get')) {
    /**
     * Translation get: checks (lang, scope) then (lang, global) then fallback default.
     */
    function cpExtAdmin_t_get(string $key, string $lang = 'english', string $scope = 'global', string $default = ''): string
    {
        try {
            $row = Capsule::table('mod_cp_ext_translations')
                ->where('lang', $lang)
                ->where('scope', $scope)
                ->where('key', $key)
                ->first();
            if ($row && $row->value !== null && $row->value !== '') {
                return (string)$row->value;
            }
            if ($scope !== 'global') {
                $row2 = Capsule::table('mod_cp_ext_translations')
                    ->where('lang', $lang)
                    ->where('scope', 'global')
                    ->where('key', $key)
                    ->first();
                if ($row2 && $row2->value !== null && $row2->value !== '') {
                    return (string)$row2->value;
                }
            }
        } catch (\Throwable $e) {}
        return $default;
    }
}

if (!function_exists('cpExtAdmin_t_set')) {
    /**
     * Translation set.
     */
    function cpExtAdmin_t_set(string $key, string $value, string $lang = 'english', string $scope = 'global'): void
    {
        $existing = Capsule::table('mod_cp_ext_translations')
            ->where('lang', $lang)
            ->where('scope', $scope)
            ->where('key', $key)
            ->first();
        if ($existing) {
            Capsule::table('mod_cp_ext_translations')->where('id', $existing->id)->update([
                'value' => $value,
                'updated_at' => cpExtAdmin_now(),
            ]);
        } else {
            Capsule::table('mod_cp_ext_translations')->insert([
                'lang' => $lang,
                'scope' => $scope,
                'key' => $key,
                'value' => $value,
                'created_at' => cpExtAdmin_now(),
                'updated_at' => cpExtAdmin_now(),
            ]);
        }
    }
}

if (!function_exists('cpExtAdmin_t_delete')) {
    function cpExtAdmin_t_delete(int $id): void
    {
        Capsule::table('mod_cp_ext_translations')->where('id', $id)->delete();
    }
}


if (!function_exists('cpExtAdmin_encrypt')) {
    function cpExtAdmin_encrypt(string $plain): string
    {
        if ($plain === '') return '';
        // WHMCS provides encrypt()/decrypt() helpers in many installs.
        if (function_exists('encrypt')) {
            return (string)encrypt($plain);
        }
        // Fallback: weak base64 (only if encrypt() missing) – better than plaintext, but not strong.
        return base64_encode($plain);
    }
}
if (!function_exists('cpExtAdmin_decrypt')) {
    function cpExtAdmin_decrypt(string $cipher): string
    {
        if ($cipher === '') return '';
        if (function_exists('decrypt')) {
            return (string)decrypt($cipher);
        }
        $out = base64_decode($cipher, true);
        return $out === false ? '' : $out;
    }
}
if (!function_exists('cpExtAdmin_mask')) {
    function cpExtAdmin_mask(string $s): string
    {
        if ($s === '') return '';
        if (strlen($s) <= 6) return str_repeat('•', strlen($s));
        return substr($s, 0, 2) . str_repeat('•', max(0, strlen($s)-4)) . substr($s, -2);
    }
}
