<?php
/**
 * WHMCS Addon Module: cPanel Extended Admin
 * Part 8: Tightening + secret encryption + server module skeleton
 */

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

use WHMCS\Database\Capsule;

require_once __DIR__ . '/lib/helpers.php';

function cp_extended_admin_config()
{
    return [
        "name"        => "cPanel Extended Admin",
        "description" => "Admin control plane for cPanel Extended (product mapping, feature toggles, CloudLinux policies, installers, queue, translations).",
        "author"      => "Your Company",
        "language"    => "english",
        "version"     => "1.1.0",
        "fields"      => [
            "debug" => [
                "FriendlyName" => "Debug Logging",
                "Type"         => "yesno",
                "Default"      => "off",
                "Description"  => "Enable verbose logs in module logs table",
            ],
        ],
    ];
}

function cp_extended_admin_activate()
{
    try {
        // Settings
        if (!Capsule::schema()->hasTable('mod_cp_ext_settings')) {
            Capsule::schema()->create('mod_cp_ext_settings', function ($table) {
                $table->increments('id');
                $table->string('scope', 40)->default('global'); // global | product:<id> | server:<id>
                $table->string('key', 120);
                $table->longText('value')->nullable();
                $table->timestamp('created_at')->nullable();
                $table->timestamp('updated_at')->nullable();
                $table->unique(['scope', 'key']);
            });
        }

        // Logs
        if (!Capsule::schema()->hasTable('mod_cp_ext_logs')) {
            Capsule::schema()->create('mod_cp_ext_logs', function ($table) {
                $table->increments('id');
                $table->string('level', 12)->default('info');
                $table->text('message');
                $table->longText('context_json')->nullable();
                $table->timestamp('created_at')->nullable();
                $table->index(['level', 'created_at']);
            });
        }

        // Queue (foundation)
        if (!Capsule::schema()->hasTable('mod_cp_ext_queue')) {
            Capsule::schema()->create('mod_cp_ext_queue', function ($table) {
                $table->increments('id');
                $table->string('type', 40);
                $table->string('status', 20)->default('pending');
                $table->unsignedInteger('related_user_id')->nullable();
                $table->unsignedInteger('related_service_id')->nullable();
                $table->unsignedInteger('attempts')->default(0);
                $table->timestamp('available_at')->nullable();
                $table->timestamp('locked_at')->nullable();
                $table->string('lock_token', 64)->nullable();
                $table->longText('payload_json')->nullable();
                $table->longText('result_json')->nullable();
                $table->timestamp('created_at')->nullable();
                $table->timestamp('updated_at')->nullable();
                $table->index(['status', 'available_at']);
                $table->index(['type', 'status']);
            });
        }


        // Translations (language overrides)
        if (!Capsule::schema()->hasTable('mod_cp_ext_translations')) {
            Capsule::schema()->create('mod_cp_ext_translations', function ($table) {
                $table->increments('id');
                $table->string('lang', 40)->default('english');
                $table->string('scope', 40)->default('global'); // global | product:<id>
                $table->string('key', 160);
                $table->longText('value')->nullable();
                $table->timestamp('created_at')->nullable();
                $table->timestamp('updated_at')->nullable();
                $table->unique(['lang','scope','key']);
                $table->index(['lang','scope']);
            });
        }

        // Seed defaults
        cpExtAdmin_setting_set('ui.brand_name', 'cPanel Extended', 'global');
        cpExtAdmin_setting_set('ui.accent', 'emerald', 'global');

        // Queue worker secret (used by queue_worker.php). Set only if not already set.
        $existingSecret = (string)cpExtAdmin_setting_get('queue.worker_secret', 'global', '');
        if ($existingSecret === '') {
            $secret = bin2hex(random_bytes(16));
            cpExtAdmin_setting_set('queue.worker_secret', $secret, 'global');
        }
        cpExtAdmin_log('info', 'Module activated', ['version' => '1.0.7']);

        return [
            'status'      => 'success',
            'description' => 'cPanel Extended Admin activated successfully.',
        ];
    } catch (\Throwable $e) {
        return [
            'status'      => 'error',
            'description' => 'Activation failed: ' . $e->getMessage(),
        ];
    }
}

function cp_extended_admin_deactivate()
{
    cpExtAdmin_log('info', 'Module deactivated');
    return [
        'status'      => 'success',
        'description' => 'Module deactivated. Tables were retained.',
    ];
}

function cp_extended_admin_upgrade($vars)
{
    // In Part 2 we keep schema unchanged (uses settings table for per-product config).
    return;
}

/**
 * Admin output (Addon admin area).
 */
function cp_extended_admin_output($vars)
{
    cpExtAdmin_require_admin();

    $page = isset($_GET['page']) ? preg_replace('/[^a-z0-9_\-]/i', '', (string)$_GET['page']) : 'dashboard';

    $data = [
        'module' => $vars['module'],
        'brand'  => cpExtAdmin_setting_get('ui.brand_name', 'global', 'cPanel Extended'),
        'page'   => $page,
        'baseUrl'=> cpExtAdmin_admin_base_url(),
        'token'  => function_exists('generate_token') ? generate_token('plain') : '',
        'flash'  => '',
    ];

    // --- Global Settings save ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_global_ui') {
        check_token('WHMCS.admin.default');

        $brand  = trim((string)($_POST['brand_name'] ?? ''));
        $accent = trim((string)($_POST['accent'] ?? 'emerald'));
        if ($brand === '') $brand = 'cPanel Extended';

        cpExtAdmin_setting_set('ui.brand_name', $brand, 'global');
        cpExtAdmin_setting_set('ui.accent', $accent, 'global');

        cpExtAdmin_log('info', 'Updated global UI settings', ['brand' => $brand, 'accent' => $accent]);

        $data['brand'] = $brand;
        $data['flash'] = 'Settings saved.';
        $page = 'settings';
        $data['page'] = $page;
    }

    // --- Product config save (Part 2) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_product_config') {
        check_token('WHMCS.admin.default');

        $pid = (int)($_POST['pid'] ?? 0);
        if ($pid > 0) {
            $scope = 'product:' . $pid;
            $enabled = isset($_POST['enabled']) ? '1' : '0';
            $template = trim((string)($_POST['template'] ?? 'default'));
            $installer = trim((string)($_POST['installer'] ?? 'none'));

            $features = $_POST['features'] ?? [];
            if (!is_array($features)) $features = [];
            // sanitize feature keys
            $features = array_values(array_unique(array_filter(array_map(function($f){
                return preg_replace('/[^a-z0-9_\-]/i','', (string)$f);
            }, $features))));

            cpExtAdmin_setting_set('enabled', $enabled, $scope);
            cpExtAdmin_setting_set('template', $template, $scope);
            cpExtAdmin_setting_set('installer', $installer, $scope);
            cpExtAdmin_setting_set('features_json', json_encode($features), $scope);

            cpExtAdmin_log('info', 'Saved product config', [
                'product_id' => $pid,
                'enabled' => $enabled,
                'template' => $template,
                'installer' => $installer,
                'features' => $features,
            ]);

            $data['flash'] = 'Product configuration saved.';
            $_GET['page'] = 'products';
            $_GET['pid'] = (string)$pid;
            $page = 'products';
            $data['page'] = $page;
        } else {
            $data['flash'] = 'Invalid product ID.';
        }
    }

    // --- CloudLinux policy save (Part 3) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_cloudlinux_policy') {
        check_token('WHMCS.admin.default');

        $pid = (int)($_POST['pid'] ?? 0);
        if ($pid > 0) {
            $scope = 'product:' . $pid;

            $policy = [
                'enabled' => isset($_POST['cl_enabled']) ? 1 : 0,
                'php_version' => trim((string)($_POST['php_version'] ?? 'inherit')),
                'ops' => (int)($_POST['ops'] ?? 0),
                'cpu_percent' => (int)($_POST['cpu_percent'] ?? 0),
                'cpu_cores' => (int)($_POST['cpu_cores'] ?? 0),
                'io' => (int)($_POST['io'] ?? 0),
                'iops' => (int)($_POST['iops'] ?? 0),
                'nproc' => (int)($_POST['nproc'] ?? 0),
                'pmem_mb' => (int)($_POST['pmem_mb'] ?? 0),
                'vmem_mb' => (int)($_POST['vmem_mb'] ?? 0),
                'ep' => (int)($_POST['ep'] ?? 0),
                'entry_procs' => (int)($_POST['entry_procs'] ?? 0),
                'inode_limit' => (int)($_POST['inode_limit'] ?? 0),
                'notes' => trim((string)($_POST['notes'] ?? '')),
            ];

            // Normalize negative to 0
            foreach ($policy as $k => $v) {
                if (is_int($v) && $v < 0) $policy[$k] = 0;
            }

            cpExtAdmin_setting_set('cloudlinux_policy_json', json_encode($policy), $scope);

            cpExtAdmin_log('info', 'Saved CloudLinux policy', [
                'product_id' => $pid,
                'policy' => $policy,
            ]);

            $data['flash'] = 'CloudLinux policy saved.';
            $_GET['page'] = 'cloudlinux';
            $_GET['pid'] = (string)$pid;
            $page = 'cloudlinux';
            $data['page'] = $page;
        } else {
            $data['flash'] = 'Invalid product ID.';
        }
    }


    // --- Installer configuration save (Part 4) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'save_installer_config') {
        check_token('WHMCS.admin.default');

        $pid = (int)($_POST['pid'] ?? 0);
        if ($pid > 0) {
            $scope = 'product:' . $pid;

            $installer = trim((string)($_POST['installer'] ?? 'none'));
            if (!in_array($installer, ['none','softaculous','installatron'], true)) {
                $installer = 'none';
            }

            // Global installer connection/config per product (for now stored in settings JSON)
            $config = [
                'installer' => $installer,
                'softaculous' => [
                    'api_url' => trim((string)($_POST['soft_api_url'] ?? '')),
                    'api_key' => cpExtAdmin_encrypt(trim((string)($_POST['soft_api_key'] ?? ''))),
                    'api_user' => trim((string)($_POST['soft_api_user'] ?? '')),
                ],
                'installatron' => [
                    'api_url' => trim((string)($_POST['inst_api_url'] ?? '')),
                    'api_key' => cpExtAdmin_encrypt(trim((string)($_POST['inst_api_key'] ?? ''))),
                ],
                'auto_install' => [
                    'enabled' => isset($_POST['ai_enabled']) ? 1 : 0,
                    'source'  => trim((string)($_POST['ai_source'] ?? 'admin_default')), // admin_default|client_choice
                    'default_app' => trim((string)($_POST['ai_default_app'] ?? '')),
                    'allow_custom_fields' => isset($_POST['ai_allow_custom']) ? 1 : 0,
                ],
            ];

            // Auto-install app options (key/value)
            $opts = $_POST['ai_opts'] ?? [];
            if (!is_array($opts)) $opts = [];
            $clean = [];
            foreach ($opts as $row) {
                if (!is_array($row)) continue;
                $k = preg_replace('/[^a-z0-9_\-\.]/i', '', (string)($row['k'] ?? ''));
                $v = trim((string)($row['v'] ?? ''));
                if ($k === '') continue;
                $clean[$k] = $v;
            }
            $config['auto_install']['options'] = $clean;

            cpExtAdmin_setting_set('installer_config_json', json_encode($config), $scope);

            cpExtAdmin_log('info', 'Saved installer config', [
                'product_id' => $pid,
                'installer' => $installer,
                'auto_install_enabled' => $config['auto_install']['enabled'],
                'auto_install_source' => $config['auto_install']['source'],
                'default_app' => $config['auto_install']['default_app'],
            ]);

            $data['flash'] = 'Installer configuration saved.';
            $_GET['page'] = 'installers';
            $_GET['pid'] = (string)$pid;
            $page = 'installers';
            $data['page'] = $page;
        } else {
            $data['flash'] = 'Invalid product ID.';
        }
    }


    // --- Queue actions (Part 5) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'queue_retry') {
        check_token('WHMCS.admin.default');
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            Capsule::table('mod_cp_ext_queue')->where('id',$id)->update([
                'status' => 'pending',
                'locked_at' => null,
                'lock_token' => null,
                'updated_at' => cpExtAdmin_now(),
            ]);
            cpExtAdmin_log('info', 'Queue task retried', ['task_id'=>$id]);
            $data['flash'] = 'Task queued for retry.';
        }
        $_GET['page'] = 'queue';
        $page = 'queue';
        $data['page'] = $page;
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'queue_cancel') {
        check_token('WHMCS.admin.default');
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            Capsule::table('mod_cp_ext_queue')->where('id',$id)->update([
                'status' => 'cancelled',
                'locked_at' => null,
                'lock_token' => null,
                'updated_at' => cpExtAdmin_now(),
            ]);
            cpExtAdmin_log('info', 'Queue task cancelled', ['task_id'=>$id]);
            $data['flash'] = 'Task cancelled.';
        }
        $_GET['page'] = 'queue';
        $page = 'queue';
        $data['page'] = $page;
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'queue_enqueue_test') {
        check_token('WHMCS.admin.default');
        $taskId = cpExtAdmin_queue_enqueue('test_task', ['hello'=>'world']);
        cpExtAdmin_log('info', 'Enqueued test task', ['task_id'=>$taskId]);
        $data['flash'] = 'Test task enqueued (ID: ' . $taskId . ').';
        $_GET['page'] = 'queue';
        $page = 'queue';
        $data['page'] = $page;
    }


    // --- Translations tool actions (Part 6) ---
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'translation_save') {
        check_token('WHMCS.admin.default');

        $lang  = preg_replace('/[^a-z0-9_\-]/i','', (string)($_POST['lang'] ?? 'english'));
        if ($lang === '') $lang = 'english';
        $scope = preg_replace('/[^a-z0-9:\_\-]/i','', (string)($_POST['scope'] ?? 'global'));
        if ($scope === '') $scope = 'global';

        $key = trim((string)($_POST['key'] ?? ''));
        $key = preg_replace('/[^a-z0-9\.\_\-]/i','', $key);
        $val = trim((string)($_POST['value'] ?? ''));

        if ($key !== '') {
            cpExtAdmin_t_set($key, $val, $lang, $scope);
            cpExtAdmin_log('info', 'Saved translation', ['lang'=>$lang,'scope'=>$scope,'key'=>$key]);
            $data['flash'] = 'Translation saved.';
        } else {
            $data['flash'] = 'Translation key is required.';
        }
        $_GET['page'] = 'translations';
        $page = 'translations';
        $data['page'] = $page;
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'translation_delete') {
        check_token('WHMCS.admin.default');
        $id = (int)($_POST['id'] ?? 0);
        if ($id > 0) {
            cpExtAdmin_t_delete($id);
            cpExtAdmin_log('info', 'Deleted translation', ['id'=>$id]);
            $data['flash'] = 'Translation deleted.';
        }
        $_GET['page'] = 'translations';
        $page = 'translations';
        $data['page'] = $page;
    }


    // Router -> include template file
    $templateFile = __DIR__ . '/templates/admin/' . $page . '.php';
    if (!file_exists($templateFile)) {
        $templateFile = __DIR__ . '/templates/admin/404.php';
        $data['page'] = '404';
    }

    include __DIR__ . '/templates/admin/layout.php';
}

/**
 * Optional: client area output for addon module.
 * We'll implement in Part 7.
 */
function cp_extended_admin_clientarea($vars)
{
    // Client area shell for module (Part 7).
    // NOTE: The main cPanel Extended UI will live in the SERVER MODULE client area.
    // This addon client area is optional; we provide it as a branded hub / documentation / shortcuts.
    $uid = (int)($_SESSION['uid'] ?? 0);
    $lang = isset($_SESSION['Language']) ? preg_replace('/[^a-z0-9_\-]/i','', (string)$_SESSION['Language']) : 'english';
    if ($lang === '') $lang = 'english';

    $brand = cpExtAdmin_setting_get('ui.brand_name', 'global', 'cPanel Extended');

    $services = [];
    try {
        $rows = \WHMCS\Database\Capsule::table('tblhosting')
            ->join('tblproducts', 'tblproducts.id', '=', 'tblhosting.packageid')
            ->where('tblhosting.userid', $uid)
            ->whereIn('tblhosting.domainstatus', ['Active','Suspended'])
            ->select('tblhosting.id as serviceid', 'tblhosting.domain', 'tblhosting.domainstatus', 'tblproducts.id as productid', 'tblproducts.name as productname')
            ->orderBy('tblhosting.id','desc')
            ->limit(50)
            ->get();

        foreach ($rows as $r) {
            $scope = 'product:' . (int)$r->productid;
            $enabled = cpExtAdmin_setting_get('enabled', $scope, '0') === '1';
            $features = json_decode((string)cpExtAdmin_setting_get('features_json', $scope, '[]'), true);
            if (!is_array($features)) $features = [];
            $services[] = [
                'serviceid' => (int)$r->serviceid,
                'domain' => (string)$r->domain,
                'status' => (string)$r->domainstatus,
                'productid' => (int)$r->productid,
                'productname' => (string)$r->productname,
                'extended_enabled' => $enabled,
                'features' => $features,
            ];
        }
    } catch (\Throwable $e) {
        // ignore
    }

    $strings = [
        'title' => cpExtAdmin_t_get('client.title', $lang, 'global', 'cPanel Extended'),
        'subtitle' => cpExtAdmin_t_get('client.subtitle', $lang, 'global', 'Manage your hosting features directly from WHMCS.'),
        'service' => cpExtAdmin_t_get('client.service', $lang, 'global', 'Service'),
        'open' => cpExtAdmin_t_get('client.open', $lang, 'global', 'Open'),
        'disabled' => cpExtAdmin_t_get('client.disabled', $lang, 'global', 'Disabled'),
        'no_services' => cpExtAdmin_t_get('client.no_services', $lang, 'global', 'No active hosting services found.'),
        'note' => cpExtAdmin_t_get('client.note', $lang, 'global', 'Tip: The full Extended panel will appear inside each hosting service once the Server Module is installed.'),
    ];

    return [
        'pagetitle' => $brand,
        'templatefile' => 'client/hub',
        'requirelogin' => true,
        'vars' => [
            'brand' => $brand,
            'lang' => $lang,
            'strings' => $strings,
            'services' => $services,
        ],
    ];
}
